"use client"
import { useEffect, useState } from "react"

import { createClient } from "@/lib/supabase/client"
import { useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Trash2, Calendar, Clock, MapPin } from "lucide-react"
import BookingModal from "@/components/booking-modal"
import { currencyFormat } from "@/utils/common"

export interface Appointment {
  id: string
  appointment_date: string
  appointment_time: string
  status: string
  services: { name: string; price: number } | null
  branches: { name: string; address: string; city: string } | null
}

interface Branch {
  id: string
  name: string
}

interface Service {
  id: string
  name: string
  price: number
}

export default function AppointmentsPage() {
  const router = useRouter()
  const [user, setUser] = useState<any>(null)
  const [appointments, setAppointments] = useState<Appointment[]>([])
  const [loading, setLoading] = useState(true)
  const [bookingOpen, setBookingOpen] = useState(false)

  useEffect(() => {
    const fetchData = async () => {
      try {
        const supabase = createClient()
        const {
          data: { user },
          error: userError,
        } = await supabase.auth.getUser()

        if (userError || !user) {
          router.push("/auth/login")
          return
        }

        setUser(user)

        // Fetch appointments
        const { data: appointmentsData } = await supabase
          .from("appointments")
          .select(
            "id, appointment_date, appointment_time, status, services(name, price), branches(name, address, city)",
          )
          .eq("user_id", user.id)
          .order("appointment_date", { ascending: true })

        setAppointments(appointmentsData as unknown as Appointment[])
      } catch (error) {
        console.error("Error fetching data:", error)
      } finally {
        setLoading(false)
      }
    }

    fetchData()
  }, [router])

  const handleCancelAppointment = async (appointmentId: string) => {
    try {
      const supabase = createClient()
      const { error } = await supabase.from("appointments").delete().eq("id", appointmentId)

      if (error) throw error

      setAppointments(appointments.filter((a) => a.id !== appointmentId))
    } catch (error) {
      console.error("Error canceling appointment:", error)
    }
  }

  if (loading)
    return (
      <div className="flex items-center justify-center min-h-screen">
        <p className="text-muted-foreground">Loading...</p>
      </div>
    )

  return (
    <div className="min-h-screen bg-background dark:bg-background py-12 px-4">
      <div className="max-w-6xl mx-auto">
        <div className="flex items-center justify-between mb-8">
          <h1 className="text-4xl font-bold text-gold">My Appointments</h1>
          <Button onClick={() => setBookingOpen(true)} className="bg-gold text-black hover:bg-gold-dark">
            Book New Appointment
          </Button>
        </div>

        <div className="space-y-4">
          {appointments?.length === 0 ? (
            <Card className="border-border text-center py-12">
              <p className="text-muted-foreground mb-4">No appointments booked yet.</p>
              <Button onClick={() => setBookingOpen(true)} className="bg-gold text-black hover:bg-gold-dark">
                Book Your First Appointment
              </Button>
            </Card>
          ) : (
            appointments?.map((appointment) => (
              <Card key={appointment.id} className="border-border hover:border-gold/30 transition">
                <CardContent className="pt-6">
                  <div className="grid md:grid-cols-2 gap-6">
                    <div className="space-y-3">
                      <p className="font-bold text-lg text-gold">{appointment.services?.name}</p>
                      <div className="flex items-center gap-2 text-foreground">
                        <MapPin className="w-4 h-4 text-gold" />
                        <div>
                          <p className="font-semibold">{appointment.branches?.name}</p>
                          <p className="text-sm text-muted-foreground">
                            {appointment.branches?.address}, {appointment.branches?.city}
                          </p>
                        </div>
                      </div>
                    </div>
                    <div className="space-y-3">
                      <div className="flex items-center gap-2 text-foreground">
                        <Calendar className="w-4 h-4 text-gold" />
                        <p>
                          <span className="font-semibold">Date:</span>{" "}
                          {new Date(appointment.appointment_date).toLocaleDateString()}
                        </p>
                      </div>
                      <div className="flex items-center gap-2 text-foreground">
                        <Clock className="w-4 h-4 text-gold" />
                        <p>
                          <span className="font-semibold">Time:</span> {appointment.appointment_time}
                        </p>
                      </div>
                      <p className="text-foreground">
                        <span className="font-semibold">Price:</span> {currencyFormat(appointment.services?.price || 0)}
                      </p>
                    </div>
                  </div>

                  <div className="mt-6 flex items-center justify-between pt-6 border-t border-border">
                    <span className="inline-block px-3 py-1 rounded-full text-sm font-medium bg-gold/10 text-gold capitalize">
                      {appointment.status}
                    </span>
                    <button
                      onClick={() => handleCancelAppointment(appointment.id)}
                      className="flex items-center gap-2 bg-destructive hover:bg-red-700 text-white py-2 px-4 rounded-lg font-semibold transition-colors"
                    >
                      <Trash2 className="w-4 h-4" />
                      Cancel
                    </button>
                  </div>
                </CardContent>
              </Card>
            ))
          )}
        </div>
      </div>

      <BookingModal open={bookingOpen} onOpenChange={setBookingOpen} setAppointments={setAppointments} />
    </div>
  )
}
