"use client"

import { useEffect, useState } from "react"
import { motion } from "framer-motion"
import { MapPin, CheckCircle2 } from "lucide-react"
import { useBranches, Branch } from "@/context/branch-context"

export default function About() {
  const { branches } = useBranches()
  const [cities, setCities] = useState<Record<string, string[]>>({})

  useEffect(() => {
    if (branches.length > 0) {
      const grouped = branches.reduce((acc: Record<string, string[]>, branch: Branch) => {
        if (!acc[branch.city]) acc[branch.city] = []
        acc[branch.city].push(branch.name)
        return acc
      }, {})
      setCities(grouped)
    }
  }, [branches])

  return (
    <section id="about" className="py-24 px-4 sm:px-6 lg:px-8 bg-secondary/50 relative overflow-hidden">
      <div className="absolute top-0 right-0 w-1/3 h-full bg-gold/5 blur-3xl rounded-full -translate-x-1/2 -z-10" />

      <div className="max-w-7xl mx-auto">
        <div className="grid lg:grid-cols-2 gap-16 items-start">
          <motion.div
            className="space-y-8"
            initial={{ opacity: 0, x: -30 }}
            whileInView={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
          >
            <div className="space-y-4">
              <h2 className="text-4xl md:text-6xl font-extrabold text-foreground tracking-tight">
                About <span className="text-gold">United Nails</span>
              </h2>
              <p className="text-xl text-muted-foreground leading-relaxed">
                United Nails is a premier chain of luxury nail and beauty salons in India. We are dedicated to providing affordable luxury under one roof with uncompromised quality.
              </p>
            </div>

            <div className="grid sm:grid-cols-2 gap-6">
              <div className="space-y-4">
                <h3 className="text-xl font-bold text-foreground border-l-4 border-gold pl-4">Premium Services</h3>
                <ul className="space-y-3">
                  {[
                    "Intricate Nail Art & Extensions",
                    "Gel & Acrylic Enhancements",
                    "Spa Manicures & Pedicures",
                    "Eyelash Extensions",
                    "Hair Coloring & Treatments",
                    "Advanced Facials & Skin Care"
                  ].map((service, i) => (
                    <li key={i} className="flex items-center gap-2 text-muted-foreground">
                      <CheckCircle2 className="w-4 h-4 text-gold flex-shrink-0" />
                      <span className="text-sm font-medium">{service}</span>
                    </li>
                  ))}
                </ul>
              </div>

              <div className="space-y-4">
                <h3 className="text-xl font-bold text-foreground border-l-4 border-gold pl-4">Our Locations</h3>
                <div className="space-y-4">
                  {Object.keys(cities).length > 0 ? (
                    Object.entries(cities).map(([city, branches]) => (
                      <div key={city} className="flex items-start gap-3 text-muted-foreground text-sm">
                        <MapPin className="w-5 h-5 text-gold flex-shrink-0 mt-0.5" />
                        <div>
                          <p className="font-bold text-foreground text-base capitalize">{city}</p>
                          <p className="text-muted-foreground">{branches.join(" • ")}</p>
                        </div>
                      </div>
                    ))
                  ) : (
                    <div className="flex items-center gap-2 text-muted-foreground text-sm">
                      <MapPin className="w-5 h-5 text-gold animate-pulse" />
                      <span>Loading locations...</span>
                    </div>
                  )}
                </div>
              </div>
            </div>

            <div className="p-6 bg-card border border-border/50 rounded-2xl shadow-xl shadow-gold/5">
              <h3 className="text-lg font-bold text-foreground mb-2">The United Difference</h3>
              <p className="text-muted-foreground text-sm leading-relaxed italic">
                "We focus on personalized care with an uncompromised emphasis on premium products, hygiene protocols, and elite craftsmanship to deliver a fun, relaxing, and professional experience."
              </p>
            </div>
          </motion.div>

          <motion.div
            className="relative space-y-6"
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
          >
            <div className="relative h-[600px] rounded-3xl overflow-hidden shadow-2xl border-8 border-background/50 group">
              <img
                src="/images/interiror.webp"
                alt="United Nails Luxury Interior"
                className="w-full h-full object-cover transition-transform duration-1000 group-hover:scale-105"
              />
              <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent"></div>
              <div className="absolute bottom-8 left-8 right-8 grid grid-cols-3 gap-4">
                {[
                  { val: "Expert", lab: "Technicians" },
                  { lab: "Hygienic", val: "100%" },
                  { val: "Elite", lab: "Products" }
                ].map((stat, i) => (
                  <div key={i} className="text-center bg-background/20 backdrop-blur-md rounded-xl p-3 border border-white/20">
                    <p className="text-xl font-bold text-gold">{stat.val}</p>
                    <p className="text-[10px] uppercase tracking-widest text-white/80">{stat.lab}</p>
                  </div>
                ))}
              </div>
            </div>
          </motion.div>
        </div>
      </div>
    </section>
  )
}
