"use client"

import { useState } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Loader2 } from "lucide-react"

interface BranchModalProps {
    isOpen: boolean
    onOpenChange: (open: boolean) => void
    onSuccess: () => void
}

export function BranchModal({ isOpen, onOpenChange, onSuccess }: BranchModalProps) {
    const [isLoading, setIsLoading] = useState(false)
    const [formData, setFormData] = useState({
        name: "",
        address: "",
        city: "",
        postal_code: "",
        phone: ""
    })

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault()
        setIsLoading(true)
        const supabase = createClient()

        try {
            const { error } = await supabase.from("branches").insert([formData])
            if (error) throw error
            onSuccess()
            onOpenChange(false)
            setFormData({ name: "", address: "", city: "", postal_code: "", phone: "" })
        } catch (error) {
            console.error(error)
            alert("Failed to create branch")
        } finally {
            setIsLoading(false)
        }
    }

    return (
        <Dialog open={isOpen} onOpenChange={onOpenChange}>
            <DialogContent className="bg-card border-border">
                <DialogHeader>
                    <DialogTitle>Add New Branch</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4 py-4">
                    <div>
                        <Label>Branch Name</Label>
                        <Input
                            value={formData.name}
                            onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                            className="mt-2"
                            required
                        />
                    </div>
                    <div>
                        <Label>Address</Label>
                        <Input
                            value={formData.address}
                            onChange={(e) => setFormData({ ...formData, address: e.target.value })}
                            className="mt-2"
                            required
                        />
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                        <div>
                            <Label>City</Label>
                            <Input
                                value={formData.city}
                                onChange={(e) => setFormData({ ...formData, city: e.target.value })}
                                className="mt-2"
                                required
                            />
                        </div>
                        <div>
                            <Label>Postal Code</Label>
                            <Input
                                value={formData.postal_code}
                                onChange={(e) => setFormData({ ...formData, postal_code: e.target.value })}
                                className="mt-2"
                                required
                            />
                        </div>
                    </div>
                    <div>
                        <Label>Phone</Label>
                        <Input
                            value={formData.phone}
                            onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                            className="mt-2"
                            required
                        />
                    </div>
                    <Button type="submit" disabled={isLoading} className="w-full bg-gold hover:bg-gold-dark text-black mt-4">
                        {isLoading ? <><Loader2 className="w-4 h-4 mr-2 animate-spin" /> Creating...</> : "Create Branch"}
                    </Button>
                </form>
            </DialogContent>
        </Dialog>
    )
}
