"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { MapPin, Phone, Plus, Trash2 } from "lucide-react"
import { BranchModal } from "./branch-modal"

interface Branch {
    id: string
    name: string
    address: string
    city: string
    postal_code: string
    phone: string
}

export function BranchesView() {
    const [branches, setBranches] = useState<Branch[]>([])
    const [isModalOpen, setIsModalOpen] = useState(false)

    useEffect(() => {
        fetchBranches()
    }, [])

    const fetchBranches = async () => {
        const supabase = createClient()
        const { data } = await supabase.from("branches").select("*").order("name")
        setBranches(data || [])
    }

    const handleDelete = async (id: string) => {
        if (!confirm("Are you sure you want to delete this branch?")) return
        const supabase = createClient()
        try {
            await supabase.from("branches").delete().eq("id", id)
            setBranches(branches.filter(b => b.id !== id))
        } catch (error) {
            console.error(error)
        }
    }

    return (
        <div className="p-6 space-y-6">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold text-foreground">Branches</h2>
                <Button onClick={() => setIsModalOpen(true)} className="bg-gold hover:bg-gold-dark text-black">
                    <Plus className="w-4 h-4 mr-2" />
                    Add Branch
                </Button>
            </div>

            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                {branches.map((branch) => (
                    <Card key={branch.id} className="border-border hover:border-gold transition-colors relative group">
                        <div className="absolute top-4 right-4 opacity-0 group-hover:opacity-100 transition-opacity">
                            <Button
                                variant="destructive"
                                size="icon"
                                className="h-8 w-8"
                                onClick={() => handleDelete(branch.id)}
                            >
                                <Trash2 className="w-4 h-4" />
                            </Button>
                        </div>
                        <CardContent className="p-6 space-y-4">
                            <h3 className="text-xl font-bold text-foreground pr-8">{branch.name}</h3>

                            <div className="space-y-3 pt-2">
                                <div className="flex gap-3">
                                    <MapPin className="w-5 h-5 text-gold flex-shrink-0 mt-0.5" />
                                    <div>
                                        <p className="text-sm font-medium text-muted-foreground">Address</p>
                                        <p className="text-foreground">{branch.address}</p>
                                        <p className="text-foreground">{branch.city}, {branch.postal_code}</p>
                                    </div>
                                </div>

                                <div className="flex gap-3">
                                    <Phone className="w-5 h-5 text-gold flex-shrink-0 mt-0.5" />
                                    <div>
                                        <p className="text-sm font-medium text-muted-foreground">Phone</p>
                                        <p className="text-foreground">{branch.phone}</p>
                                    </div>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                ))}
            </div>

            <BranchModal
                isOpen={isModalOpen}
                onOpenChange={setIsModalOpen}
                onSuccess={fetchBranches}
            />
        </div>
    )
}
