"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Loader2 } from "lucide-react"

interface Category {
    id: string
    name: string
    created_at?: string
}

interface CategoryModalProps {
    isOpen: boolean
    onOpenChange: (open: boolean) => void
    onSuccess: () => void
    category?: Category | null
}

export function CategoryModal({ isOpen, onOpenChange, onSuccess, category }: CategoryModalProps) {
    const [isLoading, setIsLoading] = useState(false)
    const [name, setName] = useState("")

    useEffect(() => {
        if (category) {
            setName(category.name)
        } else {
            setName("")
        }
    }, [category, isOpen])

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault()
        setIsLoading(true)
        const supabase = createClient()

        try {
            if (category) {
                const { error } = await supabase
                    .from("categories")
                    .update({ name })
                    .eq("id", category.id)
                if (error) throw error
            } else {
                const { error } = await supabase.from("categories").insert([{ name }])
                if (error) throw error
            }
            onSuccess()
            onOpenChange(false)
            setName("")
        } catch (error) {
            console.error(error)
            alert(category ? "Failed to update category" : "Failed to create category")
        } finally {
            setIsLoading(false)
        }
    }

    return (
        <Dialog open={isOpen} onOpenChange={onOpenChange}>
            <DialogContent className="bg-card border-border">
                <DialogHeader>
                    <DialogTitle>{category ? "Edit Category" : "Add New Category"}</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4 py-4">
                    <div>
                        <Label>Category Name</Label>
                        <Input
                            value={name}
                            onChange={(e) => setName(e.target.value)}
                            className="mt-2"
                            required
                        />
                    </div>
                    <Button type="submit" disabled={isLoading} className="w-full bg-gold hover:bg-gold-dark text-black mt-4">
                        {isLoading ? <><Loader2 className="w-4 h-4 mr-2 animate-spin" /> {category ? "Updating..." : "Creating..."}</> : (category ? "Update Category" : "Create Category")}
                    </Button>
                </form>
            </DialogContent>
        </Dialog>
    )
}
