"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Search } from "lucide-react"

interface Profile {
    id: string
    full_name: string
    email: string
    phone: string
    created_at: string
}

export function CustomersView() {
    const [customers, setCustomers] = useState<Profile[]>([])
    const [searchQuery, setSearchQuery] = useState("")

    useEffect(() => {
        fetchCustomers()
    }, [])

    const fetchCustomers = async () => {
        const supabase = createClient()
        const { data, error } = await supabase
            .from("profiles")
            .select("*")
            .eq("is_admin", false)
            .order("full_name")

        if (error) {
            console.error("Error fetching customers:", error)
            return
        }

        setCustomers(data || [])
    }

    const filteredCustomers = customers.filter(customer =>
        customer.full_name?.toLowerCase().includes(searchQuery.toLowerCase()) ||
        customer.email?.toLowerCase().includes(searchQuery.toLowerCase())
    )

    return (
        <div className="p-6 space-y-6">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold text-foreground">Customers</h2>
                <div className="relative w-72">
                    <Search className="absolute left-3 top-2.5 h-4 w-4 text-muted-foreground" />
                    <Input
                        placeholder="Search customers..."
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                        className="pl-9 bg-input border-border"
                    />
                </div>
            </div>

            <Card className="border-border">
                <CardContent className="p-0">
                    <div className="overflow-x-auto">
                        <table className="w-full text-sm text-left">
                            <thead className="bg-secondary text-foreground uppercase border-b border-border">
                                <tr>
                                    <th className="px-6 py-3 font-semibold">Name</th>
                                    <th className="px-6 py-3 font-semibold">Email</th>
                                    <th className="px-6 py-3 font-semibold">Phone</th>
                                    <th className="px-6 py-3 font-semibold">Joined Date</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-border">
                                {filteredCustomers.length > 0 ? (
                                    filteredCustomers.map((customer) => (
                                        <tr key={customer.id} className="hover:bg-muted/50 transition-colors">
                                            <td className="px-6 py-4 font-medium text-foreground">
                                                {customer.full_name || "N/A"}
                                            </td>
                                            <td className="px-6 py-4 text-muted-foreground">
                                                {customer.email}
                                            </td>
                                            <td className="px-6 py-4 text-muted-foreground">
                                                {customer.phone || "-"}
                                            </td>
                                            <td className="px-6 py-4 text-muted-foreground">
                                                {new Date(customer.created_at).toLocaleDateString()}
                                            </td>
                                        </tr>
                                    ))
                                ) : (
                                    <tr>
                                        <td colSpan={4} className="px-6 py-8 text-center text-muted-foreground">
                                            No customers found.
                                        </td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                </CardContent>
            </Card>
        </div>
    )
}
