"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Plus, Trash2 } from "lucide-react"
import { GalleryUploadModal } from "./gallery-upload-modal"
import { Switch } from "@/components/ui/switch"
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query"

interface GalleryImage {
    id: string
    title: string
    category: string
    image_url: string
    branch_id: string | null
    branches?: { name: string, city: string } | null
    show_in_landing_page: boolean
}

interface Branch {
    id: string
    name: string
    city: string
}

export function GalleryView() {
    const [isModalOpen, setIsModalOpen] = useState(false)
    const [selectedBranch, setSelectedBranch] = useState<string>("")
    const queryClient = useQueryClient()
    const supabase = createClient()

    const { data: branches = [] } = useQuery<Branch[]>({
        queryKey: ["branches"],
        queryFn: () => Promise.resolve(
            supabase
                .from("branches")
                .select("id, name, city")
                .order("name")
                .then(({ data }) => data as Branch[] || [])
        ),
        staleTime: 1000 * 60 * 60
    })

    useEffect(() => {
        if (!selectedBranch && branches.length > 0) {
            setSelectedBranch(branches[0].id)
        }
    }, [branches, selectedBranch])

    const { data: images = [] } = useQuery<GalleryImage[]>({
        queryKey: ["images", selectedBranch],
        queryFn: () => {
            let query = supabase
                .from("gallery_images")
                .select("*, branches(name, city)")
                .order("id", { ascending: false })
            if (selectedBranch && selectedBranch !== "all") {
                query = query.eq("branch_id", selectedBranch)
            }

            return Promise.resolve(
                query.then(({ data }) => data as GalleryImage[] || [])
            )
        },
        enabled: true
    })

    const { mutate: toggleShowInLanding } = useMutation({
        mutationFn: async ({ id, checked }: { id: string; checked: boolean }) => {
            const { error } = await supabase
                .from("gallery_images")
                .update({ show_in_landing_page: checked })
                .eq("id", id)
            if (error) throw error
        },
        onMutate: async ({ id, checked }) => {
            await queryClient.cancelQueries({ queryKey: ["images", selectedBranch] })
            const previousImages = queryClient.getQueryData<GalleryImage[]>(["images", selectedBranch])

            queryClient.setQueryData<GalleryImage[]>(["images", selectedBranch], (old) =>
                old?.map(img => img.id === id ? { ...img, show_in_landing_page: checked } : img)
            )

            return { previousImages }
        },
        onError: (err, newTodo, context) => {
            queryClient.setQueryData(["images", selectedBranch], context?.previousImages)
            alert("Failed to update status")
        },
        onSettled: () => {
            queryClient.invalidateQueries({ queryKey: ["images", selectedBranch] })
        }
    })

    const { mutate: deleteImage } = useMutation({
        mutationFn: async ({ id }: { id: string }) => {
            const { error } = await supabase.from("gallery_images").delete().eq("id", id)
            if (error) throw error
        },
        onMutate: async ({ id }) => {
            await queryClient.cancelQueries({ queryKey: ["images", selectedBranch] })
            const previousImages = queryClient.getQueryData<GalleryImage[]>(["images", selectedBranch])

            queryClient.setQueryData<GalleryImage[]>(["images", selectedBranch], (old) =>
                old?.filter(img => img.id !== id)
            )

            return { previousImages }
        },
        onError: (err, newTodo, context) => {
            queryClient.setQueryData(["images", selectedBranch], context?.previousImages)
            alert("Failed to delete image")
        },
        onSettled: () => {
            queryClient.invalidateQueries({ queryKey: ["images", selectedBranch] })
        }
    })

    const handleDelete = (id: string) => {
        if (confirm("Are you sure you want to delete this image?")) {
            deleteImage({ id })
        }
    }

    const handleUploadComplete = () => {
        queryClient.invalidateQueries({ queryKey: ["images"] })
    }

    return (
        <div className="p-6 space-y-6">
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                <div className="flex items-center gap-4">
                    <h2 className="text-2xl font-bold text-foreground">Gallery Management</h2>
                    <select
                        value={selectedBranch || "all"}
                        onChange={(e) => setSelectedBranch(e.target.value)}
                        className="h-10 px-3 py-2 bg-card border border-border rounded-lg text-foreground focus:outline-none focus:ring-2 focus:ring-ring"
                    >
                        <option value="all">All Branches</option>
                        {branches.map(branch => (
                            <option key={branch.id} value={branch.id}>{branch.name} ({branch.city})</option>
                        ))}
                    </select>
                </div>

                <Button
                    onClick={() => setIsModalOpen(true)}
                    className="bg-gold hover:bg-gold-dark text-black"
                >
                    <Plus className="w-4 h-4 mr-2" />
                    Add Images
                </Button>

                <GalleryUploadModal
                    isOpen={isModalOpen}
                    onOpenChange={setIsModalOpen}
                    branches={branches}
                    onUploadComplete={handleUploadComplete}
                />
            </div>

            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-6">
                {images?.map(image => (
                    <Card key={image.id} className="border-border overflow-hidden group relative">
                        <div className="aspect-square relative">
                            <img
                                src={image.image_url}
                                alt={image.title}
                                className="w-100 h-50 object-cover"
                            />
                            <div className="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                                <Button
                                    variant="destructive"
                                    size="icon"
                                    onClick={() => handleDelete(image.id)}
                                >
                                    <Trash2 className="w-5 h-5" />
                                </Button>
                            </div>
                        </div>
                        <CardContent className="p-3">
                            <div className="flex flex-col gap-2 mt-1">
                                {image.branches && (
                                    <div className="flex items-center gap-1">
                                        <span className="bg-secondary px-2 py-0.5 rounded-full text-xs text-secondary-foreground">
                                            {image.branches.name}
                                        </span>
                                        <span className="text-xs text-muted-foreground">
                                            {image.branches.city}
                                        </span>
                                    </div>
                                )}
                                <div className="flex items-center justify-between pt-3 border-t border-border mt-2">
                                    <span className="text-sm font-medium text-foreground">Show in Landing</span>
                                    <Switch
                                        checked={image.show_in_landing_page}
                                        onCheckedChange={(checked) => toggleShowInLanding({ id: image.id, checked })}
                                        className="data-[state=checked]:bg-gold data-[state=unchecked]:bg-secondary border border-muted"
                                    />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                ))}
            </div>
        </div>
    )
}
