"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Loader2, X } from "lucide-react"

interface Service {
    id: string
    name: string
    description: string | null
    sub_category_id: string
    service_prices?: ServicePrice[]
}

interface ServicePrice {
    id: string
    service_id: string
    branch_id: string
    price: number
}

interface Branch {
    id: string
    name: string
    city: string
}

interface BranchPrice {
    branch_id: string
    price: string
}

interface ServiceModalProps {
    isOpen: boolean
    onOpenChange: (open: boolean) => void
    onSuccess: () => void
    subcategoryId: string
    service?: Service | null
}

export function ServiceModal({ isOpen, onOpenChange, onSuccess, subcategoryId, service }: ServiceModalProps) {
    const [isLoading, setIsLoading] = useState(false)
    const [name, setName] = useState("")
    const [description, setDescription] = useState("")
    const [branches, setBranches] = useState<Branch[]>([])
    const [branchPrices, setBranchPrices] = useState<BranchPrice[]>([])

    useEffect(() => {
        fetchBranches()
    }, [])

    useEffect(() => {
        if (service) {
            setName(service.name)
            setDescription(service.description || "")
            if (service.service_prices) {
                setBranchPrices(
                    service.service_prices.map(sp => ({
                        branch_id: sp.branch_id,
                        price: sp.price.toString()
                    }))
                )
            }
        } else {
            setName("")
            setDescription("")
            setBranchPrices([])
        }
    }, [service, isOpen])

    const fetchBranches = async () => {
        const supabase = createClient()
        const { data } = await supabase.from("branches").select("id, name, city").order("name")
        setBranches(data || [])
    }

    const addBranchPrice = () => {
        if (branches.length > 0) {
            const availableBranches = branches.filter(
                b => !branchPrices.some(bp => bp.branch_id === b.id)
            )
            if (availableBranches.length > 0) {
                setBranchPrices([...branchPrices, { branch_id: availableBranches[0].id, price: "" }])
            }
        }
    }

    const removeBranchPrice = (index: number) => {
        setBranchPrices(branchPrices.filter((_, i) => i !== index))
    }

    const updateBranchPrice = (index: number, field: keyof BranchPrice, value: string) => {
        const updated = [...branchPrices]
        updated[index] = { ...updated[index], [field]: value }
        setBranchPrices(updated)
    }

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault()
        setIsLoading(true)
        const supabase = createClient()

        try {
            let serviceId = service?.id

            if (service) {
                const { error } = await supabase
                    .from("services")
                    .update({ name, description: description || null })
                    .eq("id", service.id)
                if (error) throw error

                await supabase.from("service_prices").delete().eq("service_id", service.id)
            } else {
                const { data, error } = await supabase
                    .from("services")
                    .insert([{ name, description: description || null, sub_category_id: subcategoryId }])
                    .select()
                    .single()
                if (error) throw error
                serviceId = data.id
            }

            if (branchPrices.length > 0 && serviceId) {
                const pricesToInsert = branchPrices
                    .filter(bp => bp.price && parseFloat(bp.price) > 0)
                    .map(bp => ({
                        service_id: serviceId,
                        branch_id: bp.branch_id,
                        price: parseFloat(bp.price)
                    }))

                if (pricesToInsert.length > 0) {
                    const { error } = await supabase.from("service_prices").insert(pricesToInsert)
                    if (error) throw error
                }
            }

            onSuccess()
            onOpenChange(false)
            setName("")
            setDescription("")
            setBranchPrices([])
        } catch (error) {
            console.error(error)
            alert(service ? "Failed to update service" : "Failed to create service")
        } finally {
            setIsLoading(false)
        }
    }

    return (
        <Dialog open={isOpen} onOpenChange={onOpenChange}>
            <DialogContent className="bg-card border-border max-w-2xl max-h-[90vh] overflow-y-auto">
                <DialogHeader>
                    <DialogTitle>{service ? "Edit Service" : "Add New Service"}</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4 py-4">
                    <div>
                        <Label>Service Name</Label>
                        <Input
                            value={name}
                            onChange={(e) => setName(e.target.value)}
                            className="mt-2"
                            required
                        />
                    </div>
                    <div>
                        <Label>Description (Optional)</Label>
                        <textarea
                            value={description}
                            onChange={(e) => setDescription(e.target.value)}
                            className="w-full mt-2 px-3 py-2 bg-input border border-border rounded-lg text-foreground"
                            rows={3}
                        />
                    </div>

                    <div className="space-y-3">
                        <div className="flex items-center justify-between">
                            <Label>Branch Pricing</Label>
                            <Button
                                type="button"
                                variant="outline"
                                size="sm"
                                onClick={addBranchPrice}
                                disabled={branchPrices.length >= branches.length}
                            >
                                Add Branch
                            </Button>
                        </div>

                        {branchPrices.map((bp, index) => (
                            <div key={index} className="flex gap-2 items-end">
                                <div className="flex-1">
                                    <Label className="text-xs">Branch</Label>
                                    <select
                                        value={bp.branch_id}
                                        onChange={(e) => updateBranchPrice(index, "branch_id", e.target.value)}
                                        className="w-full mt-1 px-3 py-2 bg-input border border-border rounded-lg text-foreground"
                                    >
                                        {branches.map(branch => (
                                            <option key={branch.id} value={branch.id}>
                                                {branch.name} ({branch.city})
                                            </option>
                                        ))}
                                    </select>
                                </div>
                                <div className="flex-1">
                                    <Label className="text-xs">Price</Label>
                                    <Input
                                        type="number"
                                        step="0.01"
                                        value={bp.price}
                                        onChange={(e) => updateBranchPrice(index, "price", e.target.value)}
                                        className="mt-1"
                                        required
                                    />
                                </div>
                                <Button
                                    type="button"
                                    variant="ghost"
                                    size="icon"
                                    onClick={() => removeBranchPrice(index)}
                                    className="hover:text-destructive"
                                >
                                    <X className="w-4 h-4" />
                                </Button>
                            </div>
                        ))}
                    </div>

                    <Button type="submit" disabled={isLoading} className="w-full bg-gold hover:bg-gold-dark text-black mt-4">
                        {isLoading ? <><Loader2 className="w-4 h-4 mr-2 animate-spin" /> {service ? "Updating..." : "Creating..."}</> : (service ? "Update Service" : "Create Service")}
                    </Button>
                </form>
            </DialogContent>
        </Dialog>
    )
}
