"use client"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { createClient } from "@/lib/supabase/client"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Plus, Pencil, Trash2 } from "lucide-react"
import { CategoryModal } from "./category-modal"
import { ConfirmationModal } from "./confirmation-modal"

interface Category {
    id: string
    name: string
    created_at: string
}

export function ServicesView() {
    const router = useRouter()
    const [categories, setCategories] = useState<Category[]>([])
    const [isModalOpen, setIsModalOpen] = useState(false)
    const [editingCategory, setEditingCategory] = useState<Category | null>(null)
    const [deletingCategory, setDeletingCategory] = useState<Category | null>(null)
    const [isDeleteLoading, setIsDeleteLoading] = useState(false)

    useEffect(() => {
        fetchCategories()
    }, [])

    const fetchCategories = async () => {
        const supabase = createClient()
        const { data } = await supabase.from("categories").select("*").order("created_at")
        setCategories(data || [])
    }

    const openCreateModal = () => {
        setEditingCategory(null)
        setIsModalOpen(true)
    }

    const handleEdit = (category: Category, e: React.MouseEvent) => {
        e.stopPropagation()
        setEditingCategory(category)
        setIsModalOpen(true)
    }

    const handleDeleteClick = (category: Category, e: React.MouseEvent) => {
        e.stopPropagation()
        setDeletingCategory(category)
    }

    const confirmDelete = async () => {
        if (!deletingCategory) return
        setIsDeleteLoading(true)
        const supabase = createClient()
        try {
            const { error } = await supabase.from("categories").delete().eq("id", deletingCategory.id)
            if (error) throw error
            await fetchCategories()
            setDeletingCategory(null)
        } catch (error) {
            console.error(error)
            alert("Failed to delete category")
        } finally {
            setIsDeleteLoading(false)
        }
    }

    const handleCategoryClick = (categoryId: string) => {
        router.push(`/admin/services/${categoryId}`)
    }

    return (
        <div className="p-6 space-y-6">
            <div className="flex items-center justify-between">
                <h2 className="text-2xl font-bold text-foreground">Categories</h2>
                <Button
                    onClick={openCreateModal}
                    className="bg-gold hover:bg-gold-dark text-black gap-2"
                >
                    Add Category <Plus className="w-4 h-4" />
                </Button>
            </div>
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                {categories.map((category) => (
                    <Card
                        key={category.id}
                        className="border-border hover:border-gold transition-colors group relative cursor-pointer"
                        onClick={() => handleCategoryClick(category.id)}
                    >
                        <CardContent className="p-6 flex items-center justify-between">
                            <span className="font-semibold text-lg text-foreground">{category.name}</span>
                            <div className="flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={(e) => handleEdit(category, e)}
                                    className="h-8 w-8 hover:text-gold"
                                >
                                    <Pencil className="w-4 h-4" />
                                </Button>
                                <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={(e) => handleDeleteClick(category, e)}
                                    className="h-8 w-8 hover:text-destructive"
                                >
                                    <Trash2 className="w-4 h-4" />
                                </Button>
                            </div>
                        </CardContent>
                    </Card>
                ))}
            </div>
            <CategoryModal
                isOpen={isModalOpen}
                onOpenChange={setIsModalOpen}
                onSuccess={fetchCategories}
                category={editingCategory}
            />
            <ConfirmationModal
                isOpen={!!deletingCategory}
                onOpenChange={(open) => !open && setDeletingCategory(null)}
                title="Delete Category"
                description={`Are you sure you want to delete "${deletingCategory?.name}"? This action cannot be undone.`}
                onConfirm={confirmDelete}
                isLoading={isDeleteLoading}
                variant="destructive"
                confirmLabel="Delete"
            />
        </div>
    )
}
