"use client"

import { useState, useEffect } from "react"
import { useParams, useRouter } from "next/navigation"
import { createClient } from "@/lib/supabase/client"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { ArrowLeft, Plus, Pencil, Trash2 } from "lucide-react"
import { SubcategoryModal } from "./subcategory-modal"
import { ConfirmationModal } from "./confirmation-modal"

interface Subcategory {
    id: string
    name: string
    category_id: string
    created_at: string
}

interface Category {
    id: string
    name: string
}

export function SubcategoryView() {
    const params = useParams()
    const router = useRouter()
    const categoryId = params.categoryId as string

    const [subcategories, setSubcategories] = useState<Subcategory[]>([])
    const [category, setCategory] = useState<Category | null>(null)
    const [isModalOpen, setIsModalOpen] = useState(false)
    const [editingSubcategory, setEditingSubcategory] = useState<Subcategory | null>(null)
    const [deletingSubcategory, setDeletingSubcategory] = useState<Subcategory | null>(null)
    const [isDeleteLoading, setIsDeleteLoading] = useState(false)

    useEffect(() => {
        fetchCategory()
        fetchSubcategories()
    }, [categoryId])

    const fetchCategory = async () => {
        const supabase = createClient()
        const { data } = await supabase
            .from("categories")
            .select("id, name")
            .eq("id", categoryId)
            .single()
        setCategory(data)
    }

    const fetchSubcategories = async () => {
        const supabase = createClient()
        const { data } = await supabase
            .from("sub_categories")
            .select("*")
            .eq("category_id", categoryId)
            .order("created_at")
        setSubcategories(data || [])
    }

    const openCreateModal = () => {
        setEditingSubcategory(null)
        setIsModalOpen(true)
    }

    const handleEdit = (subcategory: Subcategory, e: React.MouseEvent) => {
        e.stopPropagation()
        setEditingSubcategory(subcategory)
        setIsModalOpen(true)
    }

    const handleDeleteClick = (subcategory: Subcategory, e: React.MouseEvent) => {
        e.stopPropagation()
        setDeletingSubcategory(subcategory)
    }

    const confirmDelete = async () => {
        if (!deletingSubcategory) return
        setIsDeleteLoading(true)
        const supabase = createClient()
        try {
            const { error } = await supabase
                .from("sub_categories")
                .delete()
                .eq("id", deletingSubcategory.id)
            if (error) throw error
            await fetchSubcategories()
            setDeletingSubcategory(null)
        } catch (error) {
            console.error(error)
            alert("Failed to delete subcategory")
        } finally {
            setIsDeleteLoading(false)
        }
    }

    const handleSubcategoryClick = (subcategoryId: string) => {
        router.push(`/admin/services/${categoryId}/${subcategoryId}`)
    }

    return (
        <div className="p-6 space-y-6">
            <div className="flex items-center gap-4">
                <Button
                    variant="ghost"
                    size="icon"
                    onClick={() => router.push("/admin/services")}
                    className="hover:bg-secondary"
                >
                    <ArrowLeft className="w-5 h-5" />
                </Button>
                <h2 className="text-2xl font-bold text-foreground">
                    {category?.name}
                </h2>
            </div>

            <div className="flex items-center justify-between">
                <p className="text-muted-foreground">
                    Manage subcategories for {category?.name}
                </p>
                <Button onClick={openCreateModal} className="bg-gold hover:bg-gold-dark text-black gap-2">
                    Add Subcategory <Plus className="w-4 h-4" />
                </Button>
            </div>

            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                {subcategories.map((subcategory) => (
                    <Card
                        key={subcategory.id}
                        className="border-border hover:border-gold transition-colors group relative cursor-pointer"
                        onClick={() => handleSubcategoryClick(subcategory.id)}
                    >
                        <CardContent className="p-6 flex items-center justify-between">
                            <span className="font-semibold text-lg text-foreground">{subcategory.name}</span>
                            <div className="flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={(e) => handleEdit(subcategory, e)}
                                    className="h-8 w-8 hover:text-gold"
                                >
                                    <Pencil className="w-4 h-4" />
                                </Button>
                                <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={(e) => handleDeleteClick(subcategory, e)}
                                    className="h-8 w-8 hover:text-destructive"
                                >
                                    <Trash2 className="w-4 h-4" />
                                </Button>
                            </div>
                        </CardContent>
                    </Card>
                ))}
            </div>

            <SubcategoryModal
                isOpen={isModalOpen}
                onOpenChange={setIsModalOpen}
                onSuccess={fetchSubcategories}
                categoryId={categoryId}
                subcategory={editingSubcategory}
            />

            <ConfirmationModal
                isOpen={!!deletingSubcategory}
                onOpenChange={(open) => !open && setDeletingSubcategory(null)}
                title="Delete Subcategory"
                description={`Are you sure you want to delete "${deletingSubcategory?.name}"? This action cannot be undone.`}
                onConfirm={confirmDelete}
                isLoading={isDeleteLoading}
                variant="destructive"
                confirmLabel="Delete"
            />
        </div>
    )
}
