"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Loader2, Upload, X, Star } from "lucide-react"

interface Testimonial {
    id: string
    user_name: string
    user_image: string | null
    rating: number
    message: string
    branch_id: string | null
    branches?: { name: string; city: string } | null
    created_at?: string
}

interface Branch {
    id: string
    name: string
    city: string
}

interface TestimonialModalProps {
    isOpen: boolean
    onOpenChange: (open: boolean) => void
    onSuccess: () => void
    testimonial?: Testimonial | null
}

export function TestimonialModal({ isOpen, onOpenChange, onSuccess, testimonial }: TestimonialModalProps) {
    const [isLoading, setIsLoading] = useState(false)
    const [userName, setUserName] = useState("")
    const [rating, setRating] = useState(5)
    const [message, setMessage] = useState("")
    const [branchId, setBranchId] = useState<string>("")
    const [branches, setBranches] = useState<Branch[]>([])
    const [imageFile, setImageFile] = useState<File | null>(null)
    const [imagePreview, setImagePreview] = useState<string | null>(null)
    const [existingImage, setExistingImage] = useState<string | null>(null)

    useEffect(() => {
        fetchBranches()
    }, [])

    const fetchBranches = async () => {
        const supabase = createClient()
        const { data } = await supabase.from("branches").select("id, name, city").order("name")
        setBranches(data || [])
        if (data && data.length > 0 && !branchId) {
            setBranchId(data[0].id)
        }
    }

    useEffect(() => {
        if (testimonial) {
            setUserName(testimonial.user_name)
            setRating(testimonial.rating)
            setMessage(testimonial.message)
            setBranchId(testimonial.branch_id || (branches.length > 0 ? branches[0].id : ""))
            setExistingImage(testimonial.user_image)
            setImagePreview(null)
            setImageFile(null)
        } else {
            setUserName("")
            setRating(5)
            setMessage("")
            setBranchId(branches.length > 0 ? branches[0].id : "")
            setExistingImage(null)
            setImagePreview(null)
            setImageFile(null)
        }
    }, [testimonial, isOpen])

    const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0]
        if (file) {
            setImageFile(file)
            const reader = new FileReader()
            reader.onloadend = () => {
                setImagePreview(reader.result as string)
            }
            reader.readAsDataURL(file)
        }
    }

    const removeImage = () => {
        setImageFile(null)
        setImagePreview(null)
        setExistingImage(null)
    }

    const uploadImage = async (file: File): Promise<string | null> => {
        const supabase = createClient()
        const fileExt = file.name.split('.').pop()
        const fileName = `${Math.random()}.${fileExt}`
        const filePath = `testimonials/${fileName}`

        const { error: uploadError } = await supabase.storage
            .from('united-nails-uploads')
            .upload(filePath, file)

        if (uploadError) {
            console.error('Upload error:', uploadError)
            return null
        }

        const { data } = supabase.storage
            .from('united-nails-uploads')
            .getPublicUrl(filePath)

        return data.publicUrl
    }

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault()
        setIsLoading(true)
        const supabase = createClient()

        try {
            let imageUrl = existingImage

            if (imageFile) {
                const uploadedUrl = await uploadImage(imageFile)
                if (uploadedUrl) {
                    imageUrl = uploadedUrl
                }
            }

            const testimonialData = {
                user_name: userName,
                rating,
                message: message,
                user_image: imageUrl,
                branch_id: branchId || null
            }
            if (testimonial) {
                const { error } = await supabase
                    .from("testimonials")
                    .update(testimonialData)
                    .eq("id", testimonial.id)
                if (error) throw error
            } else {
                const { error } = await supabase
                    .from("testimonials")
                    .insert([testimonialData])
                if (error) throw error
            }

            onSuccess()
            onOpenChange(false)
            setUserName("")
            setRating(5)
            setMessage("")
            setBranchId(branches.length > 0 ? branches[0].id : "")
            setImageFile(null)
            setImagePreview(null)
            setExistingImage(null)
        } catch (error) {
            console.error(error)
            alert(testimonial ? "Failed to update testimonial" : "Failed to create testimonial")
        } finally {
            setIsLoading(false)
        }
    }

    return (
        <Dialog open={isOpen} onOpenChange={onOpenChange}>
            <DialogContent className="bg-card border-border max-w-2xl">
                <DialogHeader>
                    <DialogTitle>{testimonial ? "Edit Testimonial" : "Add New Testimonial"}</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4 py-4">
                    <div>
                        <Label>User Name</Label>
                        <Input
                            value={userName}
                            onChange={(e) => setUserName(e.target.value)}
                            className="mt-2"
                            required
                        />
                    </div>

                    <div>
                        <Label>Rating</Label>
                        <div className="flex gap-2 mt-2">
                            {[1, 2, 3, 4, 5].map((star) => (
                                <button
                                    key={star}
                                    type="button"
                                    onClick={() => setRating(star)}
                                    className="focus:outline-none"
                                >
                                    <Star
                                        className={`w-8 h-8 transition-colors ${star <= rating ? "fill-gold text-gold" : "text-muted-foreground hover:text-gold"
                                            }`}
                                    />
                                </button>
                            ))}
                        </div>
                    </div>

                    <div>
                        <Label>Comment</Label>
                        <textarea
                            value={message}
                            onChange={(e) => setMessage(e.target.value)}
                            className="w-full mt-2 px-3 py-2 bg-input border border-border rounded-lg text-foreground"
                            rows={4}
                            required
                        />
                    </div>

                    <div>
                        <Label>User Image (Optional)</Label>
                        <div className="mt-2">
                            {(imagePreview || existingImage) ? (
                                <div className="relative inline-block">
                                    <img
                                        src={imagePreview || existingImage || ""}
                                        alt="Preview"
                                        className="w-24 h-24 rounded-full object-cover"
                                    />
                                    <button
                                        type="button"
                                        onClick={removeImage}
                                        className="absolute -top-2 -right-2 bg-destructive text-destructive-foreground rounded-full p-1 hover:bg-destructive/90"
                                    >
                                        <X className="w-4 h-4" />
                                    </button>
                                </div>
                            ) : (
                                <label className="flex items-center justify-center w-24 h-24 border-2 border-dashed border-border rounded-full cursor-pointer hover:border-gold transition-colors">
                                    <Upload className="w-6 h-6 text-muted-foreground" />
                                    <input
                                        type="file"
                                        accept="image/*"
                                        onChange={handleImageChange}
                                        className="hidden"
                                    />
                                </label>
                            )}
                        </div>
                    </div>

                    <div>
                        <Label>Branch</Label>
                        <select
                            value={branchId}
                            onChange={(e) => setBranchId(e.target.value)}
                            className="w-full mt-2 px-3 py-2 bg-input border border-border rounded-lg text-foreground"
                            required
                        >
                            {branches.map(branch => (
                                <option key={branch.id} value={branch.id}>
                                    {branch.name} ({branch.city})
                                </option>
                            ))}
                        </select>
                    </div>

                    <Button type="submit" disabled={isLoading} className="w-full bg-gold hover:bg-gold-dark text-black mt-4">
                        {isLoading ? <><Loader2 className="w-4 h-4 mr-2 animate-spin" /> {testimonial ? "Updating..." : "Creating..."}</> : (testimonial ? "Update Testimonial" : "Create Testimonial")}
                    </Button>
                </form>
            </DialogContent>
        </Dialog>
    )
}
