"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { sendWhatsAppNotification } from "@/utils/whatsapp-helper"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Label } from "@/components/ui/label"
import { Input } from "@/components/ui/input"
import { Calendar } from "@/components/ui/calendar"
import { MapPin, Clock, Sparkles, Search, Check, User, Phone, Users } from "lucide-react"
import type { Dispatch, SetStateAction } from "react"
import { Appointment } from "@/app/appointments/page"
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip"

interface BookingModalProps {
  open: boolean
  onOpenChange: Dispatch<SetStateAction<boolean>>
  setAppointments?: Dispatch<SetStateAction<Appointment[]>>
}

interface Branch {
  id: string
  name: string
  address: string
  city: string
  phone: string
}

interface Service {
  id: string
  name: string
  category: string
  price: number
}

export default function BookingModal({ open, onOpenChange }: BookingModalProps) {
  const [step, setStep] = useState(1)
  const [branches, setBranches] = useState<Branch[]>([])
  const [services, setServices] = useState<Service[]>([])
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState("")
  const [searchQuery, setSearchQuery] = useState("")

  const [selectedBranch, setSelectedBranch] = useState("")
  const [selectedServices, setSelectedServices] = useState<string[]>([])
  const [selectedDate, setSelectedDate] = useState<Date | undefined>()
  const [selectedTime, setSelectedTime] = useState("")

  const [customerName, setCustomerName] = useState("")
  const [customerPhone, setCustomerPhone] = useState("")
  const [customerGender, setCustomerGender] = useState("")

  const timeSlots = [
    "09:00", "09:30", "10:00", "10:30", "11:00", "11:30",
    "12:00", "12:30", "14:00", "14:30", "15:00", "15:30",
    "16:00", "16:30", "17:00", "17:30", "18:00", "18:30",
    "19:00", "19:30", "20:00"
  ]

  const formatTo12Hr = (time24: string) => {
    const [hours, minutes] = time24.split(":").map(Number)
    const period = hours >= 12 ? "PM" : "AM"
    const hours12 = hours % 12 || 12
    return `${hours12}:${minutes.toString().padStart(2, "0")} ${period}`
  }

  const getAvailableTimeSlots = () => {
    if (!selectedDate) return timeSlots

    const today = new Date()
    const isToday = selectedDate.toDateString() === today.toDateString()

    if (!isToday) return timeSlots

    const currentHours = today.getHours()
    const currentMinutes = today.getMinutes()

    return timeSlots.filter(time => {
      const [hours, minutes] = time.split(":").map(Number)
      return hours > currentHours || (hours === currentHours && minutes > currentMinutes)
    })
  }

  useEffect(() => {
    if (!open) return

    const fetchData = async () => {
      setSelectedDate(new Date())
      try {
        const supabase = createClient()
        const { data: branchesData, error: branchesError } = await supabase
          .from("branches")
          .select("id, name, address, city, phone")

        if (branchesError) throw branchesError
        setBranches(branchesData || [])

        fetchServices("")
      } catch (err) {
        setError("Failed to load booking data. Please try again.")
      }
    }

    fetchData()
  }, [open])

  const fetchServices = async (query: string) => {
    try {
      const supabase = createClient()
      let request = supabase.from("services").select("*")

      if (query) {
        request = request.ilike("name", `%${query}%`)
      }

      const { data, error } = await request
      if (error) throw error
      setServices(data || [])
    } catch (err) {
      console.error(err)
    }
  }

  useEffect(() => {
    if (step === 2) {
      const timer = setTimeout(() => {
        fetchServices(searchQuery)
      }, 300)
      return () => clearTimeout(timer)
    }
  }, [searchQuery, step])

  const toggleService = (id: string) => {
    setSelectedServices(prev =>
      prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]
    )
  }

  const resetForm = () => {
    setSelectedBranch("")
    setSelectedServices([])
    setSelectedTime("")
    setCustomerName("")
    setCustomerPhone("")
    setCustomerGender("")
    setSelectedDate(new Date())
    setError("")
    setSearchQuery("")
    setStep(1)
  }

  const handleBooking = async () => {
    if (!selectedBranch || selectedServices.length === 0 || !selectedDate || !selectedTime || !customerName || !customerPhone || !customerGender) {
      setError("Please fill in all fields")
      return
    }

    setLoading(true)
    setError("")

    try {
      const dateString = selectedDate.toISOString().split("T")[0]

      try {
        const branch = branches.find(b => b.id === selectedBranch)
        const selectedServicesNames = selectedServices.map(id => services.find(s => s.id === id)?.name).join(", ")

        const result = await sendWhatsAppNotification({
          branchPhone: `+918347331238`,
          customerDetails: {
            name: customerName,
            phone: customerPhone,
            gender: customerGender
          },
          appointmentDetails: {
            serviceName: selectedServicesNames,
            branchName: branch?.name || "",
            date: dateString,
            time: selectedTime
          }
        })
        if (!result.success) {
          console.error("WhatsApp notification failed:", result.error)
        } else {
          resetForm()
          onOpenChange(false)
        }
      } catch (wsError) {
        console.error("WhatsApp notification failed (catch block):", wsError)
      }
    } catch (err: any) {
      setError(err.message || "Failed to book appointment. Please try again.")
    } finally {
      setLoading(false)
    }
  }

  const isStepComplete = () => {
    switch (step) {
      case 1: return selectedBranch !== ""
      case 2: return selectedServices.length > 0
      case 3: return selectedDate && selectedTime
      case 4: return customerName.trim().length > 2 && customerPhone.length > 8 && customerGender !== ""
      case 5: return true
      default: return false
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="text-2xl text-gold flex items-center gap-2">
            <Sparkles className="w-6 h-6" />
            Book Your Appointment
          </DialogTitle>
          <DialogDescription>
            Step {step} of 5: {
              step === 1 ? "Choose which of our branches you'd like to visit" :
                step === 2 ? "Select one or more services you want to book" :
                  step === 3 ? "Pick your preferred date and time" :
                    step === 4 ? "Enter your contact details" :
                      "Review and confirm your appointment"
            }
          </DialogDescription>
        </DialogHeader>

        {error && (
          <div className="bg-red-500/10 border border-red-500/20 text-red-600 px-4 py-3 rounded-lg text-sm">
            {error}
          </div>
        )}

        {step === 1 && (
          <div className="space-y-4">
            <Label className="text-base font-semibold">Select a Branch</Label>
            <div className="grid gap-3 max-h-96 overflow-y-auto pr-2">
              {branches.map((branch) => (
                <div
                  key={branch.id}
                  onClick={() => setSelectedBranch(branch.id)}
                  className={`p-4 border-2 rounded-lg cursor-pointer transition ${selectedBranch === branch.id ? "border-gold bg-gold/5" : "border-border hover:border-gold/30"}`}
                >
                  <div className="flex items-start gap-3">
                    <MapPin className="w-5 h-5 text-gold mt-0.5" />
                    <div>
                      <p className="font-semibold">{branch.city}</p>
                      <p className="text-sm text-muted-foreground">{branch.name},{branch.address} </p>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

        {step === 2 && (
          <div className="space-y-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <Input
                placeholder="Search services..."
                className="pl-9"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
              />
            </div>
            <Label className="text-sm font-medium text-muted-foreground">Select one or more services</Label>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3 max-h-80 overflow-y-auto pr-2">
              {services.map((service) => (
                <div
                  key={service.id}
                  onClick={() => toggleService(service.id)}
                  className={`p-3 border-2 rounded-lg cursor-pointer transition flex justify-between items-center ${selectedServices.includes(service.id) ? "border-gold bg-gold/5" : "border-border hover:border-gold/30"}`}
                >
                  <div className="min-w-0 pr-2">
                    <TooltipProvider>
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <p className="text-sm font-semibold truncate cursor-help">{service.name}</p>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p>{service.name}</p>
                        </TooltipContent>
                      </Tooltip>
                    </TooltipProvider>
                    <p className="text-[10px] text-gold uppercase tracking-wider">{service.category}</p>
                  </div>
                  {selectedServices.includes(service.id) && (
                    <div className="flex-shrink-0 w-5 h-5 bg-gold rounded-full flex items-center justify-center">
                      <Check className="w-3 h-3 text-black" />
                    </div>
                  )}
                </div>
              ))}
            </div>
          </div>
        )}

        {step === 3 && (
          <div className="space-y-6">
            <div>
              <Label className="text-base font-semibold mb-4 block">Select Date</Label>
              <Calendar
                mode="single"
                selected={selectedDate}
                onSelect={setSelectedDate}
                disabled={(date) => {
                  const today = new Date()
                  today.setHours(0, 0, 0, 0)
                  return date < today
                }}
                className="rounded-lg border border-border p-4"
              />
            </div>
            <div>
              <Label className="text-base font-semibold mb-4 block">Select Time</Label>
              <div className="grid grid-cols-4 gap-2">
                {getAvailableTimeSlots()?.map((time) => (
                  <button
                    key={time}
                    onClick={() => setSelectedTime(formatTo12Hr(time))}
                    className={`py-2 px-3 rounded-lg text-sm font-medium transition flex items-center justify-center gap-1 ${selectedTime === formatTo12Hr(time) ? "bg-gold text-black" : "bg-secondary border border-border hover:border-gold/30"}`}
                  >
                    <Clock className="w-4 h-4" />
                    {formatTo12Hr(time)}
                  </button>
                ))}
              </div>
            </div>
          </div>
        )}

        {step === 4 && (
          <div className="space-y-6">
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="customerName" className="flex items-center gap-2">
                  <User size={16} className="text-gold" /> Full Name
                </Label>
                <Input
                  id="customerName"
                  value={customerName}
                  onChange={(e) => setCustomerName(e.target.value)}
                  placeholder="Enter your full name"
                  className="bg-input border-border"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="customerPhone" className="flex items-center gap-2">
                  <Phone size={16} className="text-gold" /> Mobile Number
                </Label>
                <Input
                  id="customerPhone"
                  value={customerPhone}
                  onChange={(e) => setCustomerPhone(e.target.value)}
                  placeholder="Enter your mobile number"
                  className="bg-input border-border"
                />
              </div>

              <div className="space-y-2">
                <Label className="flex items-center gap-2 mb-3">
                  <Users size={16} className="text-gold" /> Gender
                </Label>
                <div className="grid grid-cols-3 gap-3">
                  {["Male", "Female", "Other"].map((gender) => (
                    <button
                      key={gender}
                      onClick={() => setCustomerGender(gender)}
                      className={`py-2 px-4 rounded-lg text-sm font-medium transition ${customerGender === gender ? "bg-gold text-black border-gold" : "bg-secondary border border-border hover:border-gold/30"} border-2`}
                    >
                      {gender}
                    </button>
                  ))}
                </div>
              </div>
            </div>
          </div>
        )}

        {step === 5 && (
          <div className="space-y-4">
            <div className="bg-secondary rounded-lg p-4 space-y-3 shadow-inner">
              <div className="flex justify-between items-start pb-2 border-b border-border">
                <span className="text-muted-foreground">Location</span>
                <span className="font-semibold">{branches.find(b => b.id === selectedBranch)?.name}</span>
              </div>
              <div className="space-y-2 pb-2 border-b border-border">
                <span className="text-muted-foreground block text-sm">Services</span>
                <div className="flex flex-wrap gap-2">
                  {selectedServices.map(id => (
                    <span key={id} className="bg-gold/10 text-gold text-xs px-2 py-1 rounded-md font-medium border border-gold/20">
                      {services.find(s => s.id === id)?.name}
                    </span>
                  ))}
                </div>
              </div>
              <div className="flex justify-between items-center pb-2 border-b border-border">
                <span className="text-muted-foreground">Date & Time</span>
                <span className="font-semibold text-sm">
                  {selectedDate?.toLocaleDateString()} at {selectedTime}
                </span>
              </div>
              <div className="space-y-2">
                <span className="text-muted-foreground block text-sm">Contact Details</span>
                <div className="flex flex-col gap-1">
                  <p className="text-sm font-semibold">{customerName}</p>
                  <p className="text-xs text-muted-foreground">{customerPhone} • {customerGender}</p>
                </div>
              </div>
            </div>
          </div>
        )}

        <div className="flex gap-3 pt-4">
          <Button
            variant="outline"
            onClick={() => step > 1 ? setStep(step - 1) : onOpenChange(false)}
            className="flex-1"
          >
            {step === 1 ? "Cancel" : "Back"}
          </Button>
          <Button
            onClick={() => step < 5 ? setStep(step + 1) : handleBooking()}
            disabled={!isStepComplete() || loading}
            className="flex-1 bg-gold text-black hover:bg-gold-dark"
          >
            {loading ? "Booking..." : step === 5 ? "Confirm Booking" : "Next"}
          </Button>
        </div>

        <div className="flex gap-2 mt-4">
          {[1, 2, 3, 4, 5].map((dot) => (
            <div key={dot} className={`h-1 flex-1 rounded-full ${dot <= step ? "bg-gold" : "bg-border"}`} />
          ))}
        </div>
      </DialogContent>
    </Dialog>
  )
}
