"use client"

import { createContext, useContext, useEffect, useState, ReactNode } from "react"
import { createClient } from "@/lib/supabase/client"

export interface Branch {
    id: string
    name: string
    address: string
    city: string
    postal_code: string
    phone: string
    images: string[]
}

interface BranchContextType {
    branches: Branch[]
    loading: boolean
    error: any
    refresh: () => Promise<void>
}

const BranchContext = createContext<BranchContextType | undefined>(undefined)

export function BranchProvider({ children }: { children: ReactNode }) {
    const [branches, setBranches] = useState<Branch[]>([])
    const [loading, setLoading] = useState(true)
    const [error, setError] = useState<any>(null)

    const fetchBranches = async () => {
        try {
            setLoading(true)
            const supabase = createClient()

            const { data: branchesData, error: branchesError } = await supabase
                .from("branches")
                .select("*")

            if (branchesError) throw branchesError

            const { data: galleryData, error: galleryError } = await supabase
                .from("gallery_images")
                .select("image_url, branch_id")
                .eq("show_in_landing_page", true)

            if (galleryError) throw galleryError

            const branchesWithImages = (branchesData || []).map((branch: any) => ({
                ...branch,
                images: (galleryData || [])
                    .filter((item: any) => item.branch_id === branch.id)
                    .map((item: any) => item.image_url)
            }))

            setBranches(branchesWithImages)
            setError(null)
        } catch (err) {
            console.error("Error in BranchContext:", err)
            setError(err)
        } finally {
            setLoading(false)
        }
    }

    useEffect(() => {
        fetchBranches()
    }, [])

    return (
        <BranchContext.Provider value={{ branches, loading, error, refresh: fetchBranches }}>
            {children}
        </BranchContext.Provider>
    )
}

export function useBranches() {
    const context = useContext(BranchContext)
    if (context === undefined) {
        throw new Error("useBranches must be used within a BranchProvider")
    }
    return context
}
